-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles
myself, the author, to unlimited free and unrestricted use,
access, and distribution of any works related to the unauthorized
user and/or distributor.

--]]

local g_thModName = g_thBetterWinch.coreData.mod.name
THVSpec_WinchableObject = {}
THVSpec_WinchableObject.CLASS_NAME = "THVSpec_WinchableObject"
THVSpec_WinchableObject.SPEC_NAME = "thWinchableObject"
THVSpec_WinchableObject.SPEC_KEY = "spec_" .. g_thModName .. "." .. THVSpec_WinchableObject.SPEC_NAME
THVSpec_WinchableObject.XML_KEY = g_thBetterWinch.coreData.xmlKey .. ".winchableObject"
local debugFlagId = THUtils.createDebugFlagId(THVSpec_WinchableObject.CLASS_NAME)
THVSpec_WinchableObject.debugFlagId = debugFlagId
local function getSpecTable(self)
    return THUtils.call(function()
        if self ~= nil then
            return self[THVSpec_WinchableObject.SPEC_KEY]
        end
    end)
end
THVSpec_WinchableObject.getSpecTable = getSpecTable
function THVSpec_WinchableObject.initSpecialization()
    local xmlSchema = Vehicle.xmlSchema
    local xmlBasePath = "vehicle." .. THVSpec_WinchableObject.XML_KEY
    local attachNodeObjectChangePath = xmlBasePath .. ".objectChanges(?)"
    xmlSchema:register(XMLValueType.NODE_INDEX, attachNodeObjectChangePath .. "#node", "Attach point node associated with the object change(s)")
    ObjectChangeUtil.registerObjectChangeXMLPaths(xmlSchema, attachNodeObjectChangePath)
end
function THVSpec_WinchableObject.prerequisitesPresent(specializations)
    if SpecializationUtil.hasSpecialization(Attachable, specializations)
        or SpecializationUtil.hasSpecialization(AttacherJoints, specializations)
        or SpecializationUtil.hasSpecialization(TensionBeltObject, specializations)
    then
        if not SpecializationUtil.hasSpecialization(SplineVehicle, specializations) then
            return true
        end
    end
    return false
end
function THVSpec_WinchableObject.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", THVSpec_WinchableObject)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", THVSpec_WinchableObject)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", THVSpec_WinchableObject)
    SpecializationUtil.registerEventListener(vehicleType, "onDelete", THVSpec_WinchableObject)
    SpecializationUtil.registerEventListener(vehicleType, "onPostDetach", THVSpec_WinchableObject)
    SpecializationUtil.registerEventListener(vehicleType, "onLeaveVehicle", THVSpec_WinchableObject)
end
function THVSpec_WinchableObject.registerOverwrittenFunctions(vehicleType)
end
function THVSpec_WinchableObject.onLoad(self, savegame)
    THUtils.pcall(function()
        local specTable = getSpecTable(self)
        if specTable ~= nil then
            specTable.vehicle = self
            specTable.userAttachPoints = {}
            specTable.userAttachPointMapping = {}
            specTable.attachNodeObjectChanges = {}
            specTable.isAttachNodeObjectChangeActive = {}
            specTable.attachedMeshNodes = {}
            specTable.isAttachable = SpecializationUtil.hasSpecialization(Attachable, self.specializations)
                or SpecializationUtil.hasSpecialization(AttacherJoints, self.specializations)
            specTable.hasTensionBelts = SpecializationUtil.hasSpecialization(TensionBeltObject, self.specializations)
            if specTable.isAttachable then
                for componentIndex = #self.components, 1, -1 do
                    local componentInfo = self.components[componentIndex]
                    THVSpec_WinchableObject.loadAttachPoints(self, componentInfo.node)
                end
            end
        end
    end)
end
function THVSpec_WinchableObject.onPostLoad(self, savegame)
    THUtils.pcall(function()
        local specTable = getSpecTable(self)
        if specTable ~= nil then
            local attachableSpec = self.spec_attachable
            local attacherJointSpec = self.spec_attacherJoints
            local isGlobalDebugEnabled = THUtils.getIsDebugEnabled()
            local xmlFile = self.xmlFile
            local components = self.components
            local i3dMappings = self.i3dMappings
            local attachNodeObjectChanges = {}
            local function addAttachNodeObjectChangeData(pAttachNodeId, pParentNodeId)
                pParentNodeId = pParentNodeId or pAttachNodeId
                if THUtils.argIsValid(type(pAttachNodeId) == THValueType.NUMBER and pAttachNodeId > 0, "attachNodeId", pAttachNodeId)
                    and THUtils.argIsValid(type(pParentNodeId) == THValueType.NUMBER and pParentNodeId > 0, "parentNodeId", pParentNodeId)
                then
                    if not THUtils.getNodeExists(pAttachNodeId) then
                        THUtils.objectErrorMsg(self, nil, "Cannot find attach node for objectChange")
                    elseif not THUtils.getNodeExists(pParentNodeId) then
                        THUtils.objectErrorMsg(self, nil, "Cannot find parent node for objectChange")
                    else
                        local objectChangeData = attachNodeObjectChanges[pParentNodeId]
                        if objectChangeData ~= nil then
                            specTable.attachNodeObjectChanges[pAttachNodeId] = objectChangeData
                            return true
                        end
                    end
                end
                return false
            end
            if xmlFile ~= nil then
                local xmlBaseKey = "vehicle." .. THVSpec_WinchableObject.XML_KEY
                xmlFile:iterate(xmlBaseKey .. ".objectChanges", function(_, pObjectChangeKey)
                    local attachNodeId = xmlFile:getValue(pObjectChangeKey .. "#node", nil, components, i3dMappings)
                    if attachNodeId ~= nil then
                        if not THUtils.getNodeExists(attachNodeId) then
                            THUtils.objectXMLErrorMsg(self, pObjectChangeKey, nil, "Cannot find object changes node")
                        else
                            local objectChangeData = {}
                            ObjectChangeUtil.loadObjectChangeFromXML(xmlFile, pObjectChangeKey, objectChangeData, components, self)
                            attachNodeObjectChanges[attachNodeId] = objectChangeData
                        end
                    end
                end)
            end
            local numUserAttachPoints = #specTable.userAttachPoints
            if numUserAttachPoints > 0 then
                for attachPointIndex = 1, numUserAttachPoints do
                    local attachNodeId = specTable.userAttachPoints[attachPointIndex]
                    if THUtils.getNodeExists(attachNodeId) then
                        local attachPointData = nil
                        local attachTypeText = "Added"
                        if getHasCollision(attachNodeId) or getHasClassId(attachNodeId, ClassIds.SHAPE) then
                            attachPointData = g_thBetterWinch:addObjectAttachPoint(self, attachNodeId)
                        else
                            attachPointData = g_thBetterWinch:injectObjectAttachPoint(self, attachNodeId, 0, 0, 0, 0, 0, 0)
                            attachTypeText = "Injected"
                        end
                        if attachPointData ~= nil then
                            if isGlobalDebugEnabled then
                                THUtils.objectDisplayMsg(self, "%s user attachment point", attachTypeText)
                                THUtils.printTable(attachPointData)
                            end
                            addAttachNodeObjectChangeData(attachPointData.node, attachNodeId)
                        end
                    end
                end
            end
            if specTable.isAttachable then
                if attachableSpec ~= nil then
                    if self.getInputAttacherJoints ~= nil then
                        local inputAttacherJoints = self:getInputAttacherJoints()
                        if inputAttacherJoints ~= nil then
                            for jointIndex = 1, #inputAttacherJoints do
                                local jointInfo = inputAttacherJoints[jointIndex]
                                if jointInfo.jointType == AttacherJoints.JOINTTYPE_TRAILER
                                    or jointInfo.jointType == AttacherJoints.JOINTTYPE_TRAILERLOW
                                    or jointInfo.jointType == AttacherJoints.JOINTTYPE_TRAILERCAR
                                then
                                    if type(jointInfo.node) == THValueType.NUMBER
                                        and jointInfo.node > 0
                                        and THUtils.getNodeExists(jointInfo.node)
                                    then
                                        local attachPointData = g_thBetterWinch:injectObjectAttachPoint(self, jointInfo.node, 0, 0, 0, 0, 0, 0)
                                        if attachPointData ~= nil then
                                            if isGlobalDebugEnabled then
                                                THUtils.objectDisplayMsg(self, "Injected attachment point to input attacher node: %s [%s]", getName(jointInfo.node), jointInfo.node)
                                                THUtils.printTable(attachPointData)
                                            end
                                            addAttachNodeObjectChangeData(attachPointData.node, jointInfo.node)
                                        end
                                    end
                                end
                            end
                        end
                    end
                end
                if attacherJointSpec ~= nil then
                    if self.getAttacherJoints ~= nil then
                        local attacherJoints = self:getAttacherJoints()
                        if attacherJoints ~= nil then
                            for jointIndex = 1, #attacherJoints do
                                local jointInfo = attacherJoints[jointIndex]
                                if jointInfo.jointType == AttacherJoints.JOINTTYPE_TRAILER
                                    or jointInfo.jointType == AttacherJoints.JOINTTYPE_TRAILERLOW
                                    or jointInfo.jointType == AttacherJoints.JOINTTYPE_TRAILERCAR
                                then
                                    if type(jointInfo.jointTransform) == THValueType.NUMBER
                                        and jointInfo.jointTransform > 0
                                        and THUtils.getNodeExists(jointInfo.jointTransform)
                                    then
                                        local attachPointData = g_thBetterWinch:injectObjectAttachPoint(self, jointInfo.jointTransform, 0, 0, 0, 0, 0, 0)
                                        if attachPointData ~= nil then
                                            if isGlobalDebugEnabled then
                                                THUtils.objectDisplayMsg(self, "Injected attachment point to attacher node: %s [%s]", getName(jointInfo.jointTransform), jointInfo.jointTransform)
                                                THUtils.printTable(attachPointData)
                                            end
                                            addAttachNodeObjectChangeData(attachPointData.node, jointInfo.jointTransform)
                                        end
                                    end
                                end
                            end
                        end
                    end
                end
            end
            if specTable.hasTensionBelts and self.getMeshNodes ~= nil then
                local tensionBeltNodes = self:getMeshNodes()
                if tensionBeltNodes ~= nil then
                    for _, tensionBeltNode in pairs(tensionBeltNodes) do
                        addAttachNodeObjectChangeData(tensionBeltNode)
                    end
                end
            end
            THUtils.clearTable(specTable.isAttachNodeObjectChangeActive)
            if next(specTable.attachNodeObjectChanges) ~= nil then
                if isGlobalDebugEnabled then
                    THUtils.objectDisplayMsg(self, "Attach node object change data:")
                end
                for attachNodeId, objectChangeData in pairs(specTable.attachNodeObjectChanges) do
                    THVSpec_WinchableObject.setAttachNodeObjectChanges(self, objectChangeData, false)
                    specTable.isAttachNodeObjectChangeActive[attachNodeId] = false
                    if isGlobalDebugEnabled then
                        THUtils.displayMsg("Node: %s [%s]", getName(attachNodeId), attachNodeId)
                        THUtils.printTable(objectChangeData, 1)
                    end
                end
            end
        end
    end)
end
function THVSpec_WinchableObject.onUpdate(self, dt)
    THUtils.pcall(function()
        THVSpec_WinchableObject.updateWinchableObject(self)
    end)
end
function THVSpec_WinchableObject.onDelete(self)
    THUtils.pcall(function()
        g_thBetterWinch:unregisterAttachableObject(self)
    end)
end
function THVSpec_WinchableObject.onPostDetach(self)
    THUtils.pcall(function()
        THVSpec_WinchableObject.setWinchableObjectDirty(self)
    end)
end
function THVSpec_WinchableObject.onLeaveVehicle(self)
    THUtils.pcall(function()
        THVSpec_WinchableObject.setWinchableObjectDirty(self)
    end)
end
function THVSpec_WinchableObject.loadAttachPoints(self, nodeId)
    local specTable = getSpecTable(self)
    if type(nodeId) == THValueType.NUMBER and nodeId > 0
        and THUtils.getNodeExists(nodeId)
    then
        if specTable.userAttachPointMapping[nodeId] == nil then
            local isAttachPoint = getUserAttribute(nodeId, "thAttachPoint")
            if isAttachPoint == true then
                table.insert(specTable.userAttachPoints, nodeId)
                specTable.userAttachPointMapping[nodeId] = #specTable.userAttachPoints
            end
        end
        local numChildren = getNumOfChildren(nodeId)
        if numChildren ~= nil and numChildren > 0 then
            for childIndex = 1, numChildren do
                local childNode = getChildAt(nodeId, childIndex - 1)
                THVSpec_WinchableObject.loadAttachPoints(self, childNode)
            end
        end
    end
end
function THVSpec_WinchableObject.setWinchableObjectDirty(self, isDirty)
    local objectData = g_thBetterWinch:getAttachableObject(self)
    if objectData ~= nil then
        objectData:setIsDirty(isDirty)
    end
end
function THVSpec_WinchableObject.updateWinchableObject(self, force)
    if THUtils.argIsValid(not force or force == true, "force", force) then
        local specTable = getSpecTable(self)
        local objectData = g_thBetterWinch:getAttachableObject(self, true)
        if objectData ~= nil then
            local _, winchRootVehicle = objectData:getActiveWinch()
            local isWinchAttached = objectData:getIsAttached() == true
            local isDebugEnabled = THUtils.getIsDebugEnabled(debugFlagId)
            if not objectData:getIsEnabled() then
                isWinchAttached = false
            end
            local rootVehicle = self
            if self.getRootVehicle ~= nil then
                rootVehicle = self:getRootVehicle() or self
            end
            local isAIActive = rootVehicle.getIsAIActive ~= nil and rootVehicle:getIsAIActive()
            local isControlled = not isAIActive and rootVehicle.getIsControlled ~= nil and rootVehicle:getIsControlled()
            local function activateVehicle(pVehicle)
                if specTable.isAttachable then
                    local attachableSpec = pVehicle.spec_attachable
                    local wheelSpec = pVehicle.spec_wheels
                    local hasAttacherVehicle = pVehicle.getAttacherVehicle ~= nil and pVehicle:getAttacherVehicle() ~= nil
                    if objectData:getIsDirty() then
                        if not hasAttacherVehicle and attachableSpec ~= nil then
                            if pVehicle.getIsSupportAnimationAllowed ~= nil
                                and attachableSpec.supportAnimations ~= nil
                            then
                                local animSpeed = -1
                                if not isWinchAttached then
                                    animSpeed = 1
                                end
                                for _, animation in pairs(attachableSpec.supportAnimations) do
                                    if pVehicle:getIsSupportAnimationAllowed(animation) then
                                        pVehicle:playAnimation(animation.animationName, animSpeed, nil, true)
                                    end
                                end
                            end
                            if wheelSpec ~= nil and wheelSpec.wheels ~= nil then
                                for _, wheel in pairs(wheelSpec.wheels) do
                                    if wheel.wheelChocks ~= nil then
                                        for _, chock in pairs(wheel.wheelChocks) do
                                            chock:update(isWinchAttached == true)
                                        end
                                    end
                                end
                            end
                        end
                    end
                    if not isControlled and not isAIActive and not hasAttacherVehicle then
                        if self.isServer and pVehicle.brake ~= nil then
                            local brakeForce = 1
                            if isWinchAttached and winchRootVehicle ~= nil then
                                brakeForce = 0.25
                                if winchRootVehicle.getLastSpeed ~= nil
                                    and winchRootVehicle.getDrivingDirection ~= nil
                                    and winchRootVehicle.getAcDecelerationAxis ~= nil
                                then
                                    local absWinchSpeed = math.abs(winchRootVehicle:getLastSpeed() or 0)
                                    local winchDirection = winchRootVehicle:getDrivingDirection() or 0
                                    if absWinchSpeed >= 1 and winchDirection ~= 0 then
                                        local winchAxisDecel = winchRootVehicle:getAcDecelerationAxis() or 0
                                        local winchBrakeFactor = THUtils.clamp(winchAxisDecel * winchDirection, -1, 1)
                                        if THUtils.getIsDebugEnabled(debugFlagId, THDebugLevel.UPDATE) then
                                            local winchName = (winchRootVehicle.getName ~= nil and winchRootVehicle:getName()) or "Winch vehicle"
                                            THUtils.displayMsg("%s:", winchName)
                                            THUtils.displayMsg("- driving speed: %s", absWinchSpeed)
                                            THUtils.displayMsg("- driving direction: %s", winchDirection)
                                            THUtils.displayMsg("- deceleration axis: %s", winchAxisDecel)
                                            THUtils.displayMsg("- brake factor: %s", winchBrakeFactor)
                                            THUtils.displayMsg("")
                                        end
                                        if pVehicle.getLastSpeed ~= nil then
                                            local absVehicleSpeed = math.abs(pVehicle:getLastSpeed() or 0)
                                            if absWinchSpeed - absVehicleSpeed <= -5 then
                                                brakeForce = 0.5
                                            end
                                        end
                                        if winchBrakeFactor < 0 then
                                            brakeForce = math.max(brakeForce, math.abs(winchBrakeFactor))
                                        end
                                    end
                                end
                            else
                                if pVehicle.getBrakeForce ~= nil then
                                    brakeForce = pVehicle:getBrakeForce()
                                    if brakeForce == nil or brakeForce < 0 then
                                        brakeForce = 1
                                    end
                                end
                            end
                            if THUtils.getIsDebugEnabled(debugFlagId, THDebugLevel.UPDATE) then
                                THUtils.objectDisplayMsg(pVehicle, "Current brake force: %s", brakeForce)
                            end
                            pVehicle:brake(brakeForce)
                        end
                    end
                    if isWinchAttached then
                        pVehicle:raiseActive()
                    end
                end
            end
            activateVehicle(rootVehicle)
            if rootVehicle.getChildVehicles ~= nil then
                local childVehicles = rootVehicle:getChildVehicles()
                if childVehicles ~= nil then
                    for _, childVehicle in pairs(childVehicles) do
                        activateVehicle(childVehicle)
                    end
                end
            end
            if objectData:getIsDirty() then
                THUtils.clearTable(specTable.attachedMeshNodes)
                for _, componentAttachPoints in pairs(objectData.componentAttachPoints) do
                    for _, attachedComponentData in pairs(componentAttachPoints) do
                        specTable.attachedMeshNodes[attachedComponentData.meshNode.node] = true
                    end
                end
                local meshNodesArray, numMeshNodes = objectData:getMeshNodes()
                if meshNodesArray ~= nil and numMeshNodes > 0 then
                    for _, meshData in pairs(meshNodesArray) do
                        local objectChangeData = specTable.attachNodeObjectChanges[meshData.node]
                        local isObjectChangeActive = specTable.isAttachNodeObjectChangeActive[meshData.node]
                        local isMeshNodeAttached = specTable.attachedMeshNodes[meshData.node] == true
                        if objectChangeData ~= nil and isObjectChangeActive ~= isMeshNodeAttached then
                            THVSpec_WinchableObject.setAttachNodeObjectChanges(self, objectChangeData, isMeshNodeAttached)
                            specTable.isAttachNodeObjectChangeActive[meshData.node] = isMeshNodeAttached
                            if isDebugEnabled then
                                if isMeshNodeAttached then
                                    THUtils.objectDisplayMsg(self, "Activating object changes for node: %s (%s)", meshData.node, meshData.nodeName)
                                else
                                    THUtils.objectDisplayMsg(self, "Deactivating object changes for node: %s (%s)", meshData.node, meshData.nodeName)
                                end
                            end
                        end
                    end
                end
                if isDebugEnabled then
                    if isWinchAttached then
                        THUtils.objectDisplayMsg(self, "Activating vehicle")
                    else
                        THUtils.objectDisplayMsg(self, "Deactivating vehicle")
                    end
                end
                objectData:setIsDirty(false)
            end
        end
    end
end
function THVSpec_WinchableObject.setAttachNodeObjectChanges(self, objectChangeData, isActive, skipMovingTools, skipInterpolation)
    local specTable = getSpecTable(self)
    THUtils.validateArg(not skipMovingTools or skipMovingTools == true, "skipMovingTools", skipMovingTools, false)
    THUtils.validateArg(not skipInterpolation or skipInterpolation == true, "skipInterpolation", skipInterpolation, false)
    if specTable ~= nil then
        if not skipMovingTools and type(self.setMovingToolDirty) == THValueType.FUNCTION then
            ObjectChangeUtil.setObjectChanges(objectChangeData, isActive, self, self.setMovingToolDirty, skipInterpolation)
        else
            ObjectChangeUtil.setObjectChanges(objectChangeData, isActive, nil, nil, skipInterpolation)
        end
    end
end