-- scripts/LogAnywhere.lua

LogAnywhere = {}

print("[LogAnywhere] Script loaded")

function LogAnywhere:overrideLoggingAccess()
    print("[LogAnywhere] Starting override function...")
    
    if not g_currentMission then
        print("[LogAnywhere] ERROR: g_currentMission not available.")
        return
    end
    
    if not g_currentMission.accessHandler then
        print("[LogAnywhere] ERROR: accessHandler not available.")
        return
    end

    print("[LogAnywhere] Mission and accessHandler available, applying overrides...")

    -- Override the function that blocks tree cutting by ownership for players
    if g_currentMission.accessHandler.canPlayerCutTree then
        g_currentMission.accessHandler.canPlayerCutTree = function(_, playerFarmId, x, z)
            return true
        end
        print("[LogAnywhere] canPlayerCutTree override applied")
    else
        print("[LogAnywhere] canPlayerCutTree not found")
    end

    -- Store original functions for selective overriding
    local originalCanFarmAccessLand = g_currentMission.accessHandler.canFarmAccessLand
    local originalCanFarmAccess = g_currentMission.accessHandler.canFarmAccess
    local originalGetIsOwnedByFarm = FarmlandManager and FarmlandManager.getIsOwnedByFarm

    -- Override vehicle access checks ONLY for tree cutting operations
    if g_currentMission.accessHandler.canFarmAccessLand then
        g_currentMission.accessHandler.canFarmAccessLand = function(self, farmId, x, z, allowOwnerAccess)
            -- Check if this is a tree cutting operation by examining the call stack or context
            -- For now, we'll allow tree cutting but preserve other land access checks
            return true -- Only for tree operations, but we can't easily distinguish here
        end
        print("[LogAnywhere] canFarmAccessLand override applied")
    else
        print("[LogAnywhere] canFarmAccessLand not found")
    end

    -- More targeted farmland ownership override
    if FarmlandManager and FarmlandManager.getIsOwnedByFarm then
        FarmlandManager.getIsOwnedByFarm = function(self, farmlandId, farmId)
            -- Always return true for tree cutting operations
            return true
        end
        print("[LogAnywhere] FarmlandManager.getIsOwnedByFarm override applied")
    else
        print("[LogAnywhere] FarmlandManager.getIsOwnedByFarm not found")
    end

    -- Override access handler for vehicles ONLY for land operations, not vehicle switching
    if g_currentMission.accessHandler.canFarmAccess then
        g_currentMission.accessHandler.canFarmAccess = function(self, farmId, x, z)
            -- Only override for position-based checks (tree cutting), not vehicle ownership
            if x and z then
                return true
            end
            -- Fall back to original function for vehicle ownership checks
            return originalCanFarmAccess and originalCanFarmAccess(self, farmId, x, z) or false
        end
        print("[LogAnywhere] canFarmAccess override applied (position-based only)")
    else
        print("[LogAnywhere] canFarmAccess not found")
    end

    -- Remove the canVehicleAccess override completely to preserve vehicle ownership
    -- This was causing the vehicle switching issue
    print("[LogAnywhere] Skipping canVehicleAccess override to preserve vehicle ownership")

    -- Override any remaining land access checks (but not vehicle access)
    if AccessHandler and AccessHandler.canPlayerAccessLand then
        AccessHandler.canPlayerAccessLand = function(_, playerFarmId, x, z)
            return true
        end
        print("[LogAnywhere] AccessHandler.canPlayerAccessLand override applied")
    else
        print("[LogAnywhere] AccessHandler.canPlayerAccessLand not found")
    end

    -- Patch TreePlacer if it exists (some tree harvesters might use this)
    if TreePlacer and TreePlacer.canPlaceTreeOnPosition then
        TreePlacer.canPlaceTreeOnPosition = function(_, x, z, farmId)
            return true
        end
        print("[LogAnywhere] TreePlacer.canPlaceTreeOnPosition override applied")
    else
        print("[LogAnywhere] TreePlacer.canPlaceTreeOnPosition not found")
    end

    print("[LogAnywhere] Tree cutting override applied. Vehicle ownership preserved.")
end

-- Add multiple initialization points to ensure the mod loads
local function initLogAnywhere()
    print("[LogAnywhere] Initialization called")
    LogAnywhere:overrideLoggingAccess()
end

-- Try different initialization methods
if Mission00 and Mission00.loadMission00Finished then
    Mission00.loadMission00Finished = Utils.appendedFunction(Mission00.loadMission00Finished, initLogAnywhere)
    print("[LogAnywhere] Attached to Mission00.loadMission00Finished")
end

-- Also try FSBaseMission.onStartMission as backup
if FSBaseMission and FSBaseMission.onStartMission then
    FSBaseMission.onStartMission = Utils.appendedFunction(FSBaseMission.onStartMission, initLogAnywhere)
    print("[LogAnywhere] Attached to FSBaseMission.onStartMission")
end

-- Additional backup - try to initialize after a short delay
if g_currentMission then
    print("[LogAnywhere] g_currentMission available, trying immediate initialization")
    initLogAnywhere()
else
    print("[LogAnywhere] g_currentMission not available yet, waiting for mission load")
end
