--[[
	PushableVehicleFix
	Specialization to fix some issues for the pushable vehicle!
	
	@author:    Ifko[nator]
	@date:      25.01.2022
	@version:	2.0

	History:	v1.0 @09.02.2021 - initial implementation in FS 19
				--------------------------------------------------
				v2.0 @25.01.2022 - convert to FS 22
]]

PushableVehicleFix = {};

--## MotorizedFix
function PushableVehicleFix:onLoad(savegame)
	local specMotorized = self.spec_motorized;
	local xmlFile = loadXMLFile("vehicle", self.xmlFile.filename);
	
	specMotorized.alwaysTurnedOn = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.motorized#alwaysTurnedOn"), string.find(self.xmlFile.filename, "FS22_lsfmFarmEquipmentPack"));

	delete(xmlFile);
end;

Motorized.onLoad = Utils.appendedFunction(Motorized.onLoad, PushableVehicleFix.onLoad);

function PushableVehicleFix:onRegisterActionEvents(superFunc, isActiveForInput, isActiveForInputIgnoreSelection)
	if self.isClient then
		local spec = self.spec_motorized;

		self:clearActionEventsTable(spec.actionEvents);

		if isActiveForInputIgnoreSelection then
			local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.TOGGLE_MOTOR_STATE, self, Motorized.actionEventToggleMotorState, false, true, false, true, nil);

			g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_VERY_HIGH);
			g_inputBinding:setActionEventText(actionEventId, spec.turnOnText);

			if (spec.motor.minForwardGearRatio == nil or spec.motor.minBackwardGearRatio == nil) and (self:getGearShiftMode() ~= VehicleMotor.SHIFT_MODE_AUTOMATIC or not GS_IS_CONSOLE_VERSION) then
				if spec.motor.manualShiftGears then
					local shiftGearInputs = {
						"SHIFT_GEAR_UP",
						"SHIFT_GEAR_DOWN"
					};

					for _, input in pairs(shiftGearInputs) do
						local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction[input], self, Motorized.actionEventShiftGear, false, true, false, true, nil);

						g_inputBinding:setActionEventTextVisibility(actionEventId, false);
					end;

					for gearNumber = 1, 8 do
						local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction["SHIFT_GEAR_SELECT_" .. gearNumber], self, Motorized.actionEventSelectGear, true, true, false, true, 1);

						g_inputBinding:setActionEventTextVisibility(actionEventId, false);
					end;
				end;

				if spec.motor.manualShiftGroups and spec.motor.gearGroups ~= nil then
					local shiftGearGroupInputs = {
						"SHIFT_GROUP_UP",
						"SHIFT_GROUP_DOWN"
					};

					for _, input in pairs(shiftGearGroupInputs) do
						local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction[input], self, Motorized.actionEventShiftGroup, false, true, false, true, nil);

						g_inputBinding:setActionEventTextVisibility(actionEventId, false);
					end;

					for groupNumber = 1, 4 do
						_, actionEventId = self:addActionEvent(spec.actionEvents, InputAction["SHIFT_GROUP_SELECT_" .. groupNumber], self, Motorized.actionEventSelectGroup, true, true, false, true, 1);

						g_inputBinding:setActionEventTextVisibility(actionEventId, false);
					end;
				end;

				_, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.AXIS_CLUTCH_VEHICLE, self, Motorized.actionEventClutch, false, false, true, true, nil);

				g_inputBinding:setActionEventTextVisibility(actionEventId, false);
			end;

			if self:getDirectionChangeMode() == VehicleMotor.DIRECTION_CHANGE_MODE_MANUAL or self:getGearShiftMode() ~= VehicleMotor.SHIFT_MODE_AUTOMATIC then
				local directionChangeInputs = {
					"DIRECTION_CHANGE",
					"DIRECTION_CHANGE_POS",
					"DIRECTION_CHANGE_NEG"
				};

				for _, input in pairs(directionChangeInputs) do
					_, actionEventId = self:addActionEvent(spec.actionEvents, InputAction[input], self, Motorized.actionEventDirectionChange, false, true, false, true, nil, nil, true);

					g_inputBinding:setActionEventTextVisibility(actionEventId, false);
				end;
			end;

			Motorized.updateActionEvents(self);
		end;
	end;
end;

Motorized.onRegisterActionEvents = Utils.overwrittenFunction(Motorized.onRegisterActionEvents, PushableVehicleFix.onRegisterActionEvents);

function PushableVehicleFix:onStateChange(superFunc, state, vehicle, isControlling)
	if state == Vehicle.STATE_CHANGE_ENTER_VEHICLE then
		if g_currentMission.missionInfo.automaticMotorStartEnabled and self:getCanMotorRun() or self.spec_motorized.alwaysTurnedOn then
			self:startMotor(true);
		end;
	elseif state == Vehicle.STATE_CHANGE_LEAVE_VEHICLE then
		if self:getStopMotorOnLeave() and g_currentMission.missionInfo.automaticMotorStartEnabled then
			self:stopMotor(true);
		end;

		self:stopVehicle();
	end;
end;

Motorized.onStateChange = Utils.overwrittenFunction(Motorized.onStateChange, PushableVehicleFix.onStateChange);